import { useEffect, useState, type ChangeEvent, type FormEvent } from 'react'
import styles from './Form.module.css'
import type { TCategory } from '../../types/TCategory'
import type { TQuote } from '../../types/TQuote'
import { mainInstance } from '../../axios/mainInstance'
import { useNavigate, useParams } from 'react-router'

type Props ={
    type: 'edit'| 'add'
}

const Form = ({type}: Props) => {
    const params = useParams()
    const [categories, setCategories] = useState<TCategory[]>([])
    const [values, setValues] =useState<TQuote>({
        author: '',
        category: '',
        text: ''
    })
    const navigate = useNavigate()

    useEffect(() => {
        const fetchCategories = async() => {
            try {
                const response = await mainInstance.get('categories.json')
                if (response?.data) {
                    setCategories(Array.from(response.data))
                }
            } catch(err) {
                console.log(err)
            }
        }
        fetchCategories()
    }, [])

    useEffect(() => {
        const fetchQuote = async () => {
            try {
                if (params?.quoteId) {
                    const response = await mainInstance.get(`quotes/${params?.quoteId}.json`)
                    if (response?.data) {
                        const temp = response.data
                        setValues({
                            text: temp.text,
                            author: temp.author,
                            category: temp.category
                        })
                    }
                }
            } catch(err) {
                console.log(err)
            }
        }
        fetchQuote()
        return () => {
            setValues({
                id: '',
                text: '',
                author: '',
                category: ''
            })
        }
    }, [params?.quoteId])

    const inputHandler = (e: ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
        setValues(prev => ({...prev, [e.target.name]: e.target.value}))
    }

    const submit = async (e: FormEvent<HTMLFormElement>) => {
        e.preventDefault()
        try {
            if (type === 'edit') {
                await mainInstance.patch(`quotes/${params.quoteId}.json`, values)
            } else {
                await mainInstance.post('quotes.json', values)
            }
            alert('Success')
            navigate('/quotes')
        } catch(err) {
            console.log(err)
        }
    }

    return (
        <section className={styles.Form__Frame}>
            <h1>{type === 'edit' ? 'Edit quote' : 'Submit new quote'}</h1>
            <form className={styles.Form} onSubmit={submit}>
                <select required value={values.category} onChange={inputHandler} name="category">
                    <option value={''} disabled>Please select category</option>
                    {categories.map(cat => {
                        return (
                            <option key={cat.value} value={cat.value}>{cat.label}</option>
                        )
                    })}
                </select>
                <input placeholder='Author' required onChange={inputHandler} value={values.author} type="text" name='author' />
                <input placeholder='Text' required onChange={inputHandler} value={values.text} type="text" name='text' />
                <button>{type === 'edit' ? 'Save edit': 'Submit'}</button>
            </form>
        </section>
    )
}

export default Form