import { countApi } from "@/api/countApi"
import { axiosInstance } from "@/axios/axios"
import { createAsyncThunk, createSlice, PayloadAction } from "@reduxjs/toolkit"

type State = {
    count: number
    loading: boolean
    error: Error | undefined
}

const namespace = 'count'
const initialState: State = {
    count: 0,
    loading: false,
    error: undefined
}

export const getCount = createAsyncThunk(
    `${namespace}/getCount`,
    async () => {
        return await countApi.getCount()
    }
)

export const putCount = createAsyncThunk(
    `${namespace}/putCount`,
    async (num: number) => {
        await countApi.putCount(num)
    }
)

const countSlice = createSlice({
    name: namespace,
    initialState,
    reducers: {
        increaseCount: (state) => {
            state.count += 1
        },
        decreaseCount: (state) => {
            state.count -= 1
        },
        plusCount: (state, action: PayloadAction<number>) => {
            state.count += action.payload
        },
        minusCount: (state, action: PayloadAction<number>) => {
            state.count -= action.payload
        },
    },
    extraReducers: (builder) => {
        builder
            .addCase(getCount.pending, (state) => {
                state.loading = true
                state.error = undefined
            })
            .addCase(getCount.rejected, (state, action) => {
                state.loading = false
                state.error = action.error as Error
            })
            .addCase(getCount.fulfilled, (state, action) => {
                state.loading = false
                state.count = action.payload
                state.error = undefined
            })
            .addCase(putCount.pending, (state) => {
                state.loading = true
                state.error = undefined
            })
            .addCase(putCount.rejected, (state) => {
                state.loading = false
                state.error = {message: 'Patch failed'} as Error
            })
            .addCase(putCount.fulfilled, (state) => {
                state.loading = false
                state.error = undefined
            })
    }
})

export const {
    decreaseCount,
    increaseCount,
    minusCount,
    plusCount,
} = countSlice.actions
export default countSlice.reducer