import { createAsyncThunk, createSlice } from "@reduxjs/toolkit"
import { axiosOrder } from "../../axios/axiosOrder"
import { TFirebaseData } from "../../types/TFirebaseData"
import { TOrder } from "../../types/TOrder"
import { AxiosRequestConfig, AxiosResponse } from "axios"

type State = {
    orders: TOrder[]
    isLoading: boolean
    error: Error | null,
}

const initialState: State = {
    orders: [],
    error: null,
    isLoading: false
}

const namespace = 'orders'

export const getOrders = createAsyncThunk(
    `${namespace}/getOrders`,
    async (): Promise<TFirebaseData<TOrder>> => {
        const response = await axiosOrder.get('orders.json')
        return response.data
    }
)

export const createOrder = createAsyncThunk(
  `${namespace}/createOrder`,
  async (payload: Omit<TOrder, 'id'>) => {
    await axiosOrder.post<AxiosRequestConfig, AxiosResponse, Omit<TOrder, 'id'>>('orders.json', payload);
  }
)

const orderSlice = createSlice({
    name: namespace,
    initialState,
    reducers: {},
    extraReducers: (builder) => {
        builder
            .addCase(getOrders.pending, (state) => {
                state.isLoading = true
                state.error = null
            })
            .addCase(getOrders.rejected, (state, action) => {
                state.isLoading = false
                state.error = action.payload as Error
            })
            .addCase(getOrders.fulfilled, (state, action) => {
                state.isLoading = false
                const fetchedOrders: TOrder[] = Object.keys(action.payload).map(id => {
                    return { ...action.payload[id] as TOrder, id };
                });
                state.orders = fetchedOrders
                state.error = null
            })
            .addCase(createOrder.pending, (state) => {
                state.isLoading = true;
                state.error = null;
            })
            .addCase(createOrder.rejected, (state, action) => {
                state.error = action.error as Error;
                state.isLoading = false
            })
            .addCase(createOrder.fulfilled, (state) => {
                state.isLoading = false;
            })
    }
})

export default orderSlice.reducer