"use client";

import * as React from "react";
import Link from "next/link";
import { useRouter } from "next/navigation";
import { useUserStore } from "@/features/userStore";
import { AppBar, Toolbar, Typography, Box, Button, Avatar, Menu, MenuItem, Divider } from "@mui/material";
import KeyboardArrowDownIcon from "@mui/icons-material/KeyboardArrowDown";
import { axiosApiClient } from "@/helpers/axiosClient";
import { useState } from "react";

export default function AppToolbar() {
    const router = useRouter();
    const { user, clearUser } = useUserStore();
    const [anchorEl, setAnchorEl] = useState<null | HTMLElement>(null);
    const open = Boolean(anchorEl);

    const handleOpen = (e: React.MouseEvent<HTMLElement>) => setAnchorEl(e.currentTarget);
    const handleClose = () => setAnchorEl(null);

    const handleProfile = () => {
        handleClose();
        router.push("/profile");
    };

    const handleLogout = async () => {
        await axiosApiClient.get("/auth/logout");
        handleClose();
        clearUser();
        router.push("/login");
    };

    return (
        <AppBar position="sticky" color="primary" sx={{ bgcolor: "#2563eb" }}>
            <Toolbar sx={{ maxWidth: 1120, mx: "auto", width: "100%" }}>
                <Typography variant="h6" component={Link} href="/" style={{ textDecoration: "none", color: "inherit" }}>
                    Computer parts shop
                </Typography>

                <Box sx={{ flexGrow: 1 }} />

                {!user ? (
                    <Button
                        component={Link}
                        href="/login"
                        color="inherit"
                        variant="outlined"
                        sx={{
                            bgcolor: "white",
                            color: "#2563eb",
                            borderColor: "#dbeafe",
                            "&:hover": { bgcolor: "#e0f2fe" },
                        }}
                    >
                        Login
                    </Button>
                ) : (
                    <>
                        <Button
                            color="inherit"
                            onClick={handleOpen}
                            aria-haspopup="menu"
                            aria-controls={open ? "user-menu" : undefined}
                            aria-expanded={open ? "true" : undefined}
                            startIcon={
                                <Avatar
                                    sx={{ width: 28, height: 28 }}
                                    src={`https://api.dicebear.com/8.x/initials/svg?seed=${encodeURIComponent(
                                        user.userName,
                                    )}`}
                                    alt={user.displayName ?? user.userName}
                                />
                            }
                            endIcon={<KeyboardArrowDownIcon />}
                            sx={{
                                textTransform: "none",
                                fontWeight: 600,
                                bgcolor: "#1d4ed8",
                                border: "1px solid #93c5fd",
                                px: 1.25,
                                "&:hover": { bgcolor: "#1e40af" },
                            }}
                        >
                            {user.displayName ?? user.userName}
                        </Button>

                        <Menu
                            id="user-menu"
                            anchorEl={anchorEl}
                            open={open}
                            onClose={handleClose}
                            transformOrigin={{ vertical: "top", horizontal: "right" }}
                            slotProps={{ paper: { sx: { mt: 1, minWidth: 200 } } }}
                        >
                            <MenuItem onClick={handleProfile}>Профиль</MenuItem>
                            <Divider />
                            <MenuItem onClick={handleLogout} sx={{ color: "error.main" }}>
                                Выйти
                            </MenuItem>
                        </Menu>
                    </>
                )}
            </Toolbar>
        </AppBar>
    );
}
