'use client'

import { Box, Button, Grid2, styled, TextField } from '@mui/material'
import { ChangeEvent, useRef, useState } from 'react'

interface IProps {
	name: string
	label: string
	onChange: (e: ChangeEvent<HTMLInputElement>) => void
	accept?: string
}

const HiddenFileInput = styled('input')({ display: 'none' })

const dragZoneStyles = {
	border: '2px dashed #a7a7a7',
	borderRadius: '8px',
	cursor: 'pointer',
}

const FileInput = ({ name, label, onChange, accept }: IProps) => {
	const inputRef = useRef<HTMLInputElement>(null)
	const [fileName, setFileName] = useState('')
	const [isDragging, setIsDragging] = useState(false)

	const openDialog = () => {
		if (inputRef?.current) {
			inputRef.current.click()
		}
	}

	const onFileChange = (e: ChangeEvent<HTMLInputElement>) => {
		const file = e.target.files[0].name
		if (file) {
			setFileName(file)
			onChange(e)
		}
	}

	const prevent = (e: DragEvent) => {
		e.preventDefault()
		e.stopPropagation()
	}

	const onDragOver = (e: any) => {
		prevent(e)
		if (!isDragging) setIsDragging(true)
	}

	const onDragLeave = (e: any) => {
		prevent(e)
		setIsDragging(false)
	}

	const onDrop = (e: any) => {
		prevent(e)
		setIsDragging(false)

		const file = e.dataTransfer?.files?.[0]
		if (!file) return

		setFileName(file.name)
	}

	return (
		<>
			<HiddenFileInput
				type='file'
				name={name}
				onChange={onFileChange}
				ref={inputRef}
				accept={accept}
			/>

			<Grid2 container direction='row' spacing={2} alignItems={'center'}>
				<Grid2>
					<Box
						sx={{
							...dragZoneStyles,
							background: isDragging ? 'gray' : 'transparent',
							borderColor: isDragging ? 'gray' : 'transparent',
						}}
						role='button'
						onDragOver={onDragOver}
						onDragLeave={onDragLeave}
						onDrop={onDrop}
					>
						<TextField
							variant='outlined'
							fullWidth
							label={label}
							value={fileName}
							disabled
							placeholder='Перетащите файл сюда или нажмите на выбор'
						/>
					</Box>
				</Grid2>

				<Grid2>
					<Button variant='contained' onClick={openDialog}>
						Browse
					</Button>
				</Grid2>
			</Grid2>
		</>
	)
}

export default FileInput
