import { Injectable, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import * as bcrypt from 'bcrypt';
import { omit } from 'lodash';
import { User } from 'src/user/entities/user.entity';
import { Repository } from 'typeorm';
import { LoginAuthDto } from './dto/login.dto';
import { RegisterAuthDto } from './dto/register.dto';

const SALT_WORK_FACTOR = 10;

@Injectable()
export class AuthService {
  constructor(
    @InjectRepository(User)
    private readonly userRepo: Repository<User>,
  ) {}

  async register(registerAuthDto: RegisterAuthDto) {
    const salt = await bcrypt.genSalt(SALT_WORK_FACTOR);

    registerAuthDto.password = await bcrypt.hash(
      registerAuthDto.password,
      salt,
    );

    const user = this.userRepo.create(registerAuthDto);

    user.generateToken();

    const userWithToken = await this.userRepo.save(user);

    // eslint-disable-next-line @typescript-eslint/no-unsafe-return, @typescript-eslint/no-unsafe-call
    return omit(userWithToken, 'password');
  }

  async login(loginAuthDto: LoginAuthDto) {
    const user = await this.userRepo.findOneBy({
      userName: loginAuthDto.userName,
    });

    if (!user) throw new NotFoundException('User not found');

    const isMatch = await user.comparePassword(loginAuthDto.password);
    if (!isMatch) throw new Error('Invalid userName or password');

    // eslint-disable-next-line @typescript-eslint/no-unsafe-return, @typescript-eslint/no-unsafe-call
    return omit(user, ['password']);
  }

  async secret(token: string) {
    const user = await this.userRepo.findOneBy({ token });
    return user;
  }

  async logout(token: string) {
    const existUser = await this.userRepo.findOneBy({ token });
    if (!existUser) throw new NotFoundException('User not found');

    existUser.token = null;

    await this.userRepo.update({ token }, existUser);
  }
}
