import {
  CanActivate,
  ExecutionContext,
  ForbiddenException,
  Injectable,
  UnauthorizedException,
} from '@nestjs/common';
import { Reflector } from '@nestjs/core';
import { Request } from 'express';
import { appDataSource } from 'src/db/dataSource';
import { ROLES_KEY } from 'src/decorators/roles.decorator';
import { User } from 'src/user/entities/user.entity';

@Injectable()
export class RolesGuard implements CanActivate {
  constructor(private reflector: Reflector) {}

  async canActivate(context: ExecutionContext): Promise<boolean> {
    const userRepo = appDataSource.getRepository(User);

    const request: Request = context.switchToHttp().getRequest();
    const authHeader = request.headers['authorization'];

    if (!authHeader) {
      throw new UnauthorizedException('Токена нет');
    }

    const requiredRoles = this.reflector.getAllAndOverride<string[]>(
      ROLES_KEY,
      [context.getHandler(), context.getClass()],
    );

    if (!requiredRoles) {
      return true; // если роли не указаны — доступ открыт
    }

    const token = authHeader;

    const user = await userRepo.findOneBy({ token });

    if (!user) {
      throw new UnauthorizedException('Юзера нет');
    }

    if (!user || !requiredRoles.includes(user?.role || '')) {
      throw new ForbiddenException('Permission denied');
    }

    return true;
  }
}
