import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import type { Ingredients, IngredientNames } from '@/interfaces/Ingredients'; 
import Burger from '@/components/Burger/Burger';
import BuildControls from '@/components/BuildControls/BuildControls';
import { IngredientPrices } from '@/helpers/IngPrice';
import { Modal } from '@/components/UI/Modal/Modal';
import { OrderSummary } from '@/components/Burger/OrderSummary';
import { useAppSelector } from '@/store';
import { useAppDispatch } from '@/store';
import { setIngredients } from '@/store/ingredients.slice';
import { getOrder } from '@/store/orders.slice';

const BurgerBuilder = () => {
    const navigate = useNavigate();
    const dispatch = useAppDispatch();
    const {ingredints} = useAppSelector(state => state);

    const [totalPrice, setTotalPrice] = useState<number>(IngredientPrices.bread);
    const [purshasable, setPurshasable] = useState<boolean>(false);
    const [show, setShow] = useState<boolean>(false);

    const onLessClick = (ingType: IngredientNames) => {
        if(ingredints[ingType] > 0) {
            dispatch(setIngredients({ingType, actionType: "less"}));
            setTotalPrice(prevState => prevState - IngredientPrices[ingType]);
            updatePurshasable(ingredints);
        }
    };

    const onMoreClick = (ingType: IngredientNames) => {
        dispatch(setIngredients({ingType, actionType: "more"}));
        setTotalPrice(prevState => prevState + IngredientPrices[ingType]);
        updatePurshasable(ingredints);
    };

    const updatePurshasable = (ings: Ingredients) => {
        const sum = Object.keys(ings) // [salad, meat, bacon, cheese]
                        .map(ing => ings[ing as IngredientNames]) // [0, 0, 0 ,0]
                        .reduce((a, b) => a + b, 0); // 0
        setPurshasable(sum > 0);
    };

    const onOpenModalHandler = () => {
        setShow(true);
    };

    const onClosedHandler = () => {
        setShow(false);
    };

    const onPurchaseContinued = () => {
        navigate({pathname: "/checkout"})
    };

    useEffect(() => {
        dispatch(getOrder())
    }, [])

    return (
        <>
            <Modal show={show} onClosed={onClosedHandler}>
                <OrderSummary 
                    ingredients={ingredints} 
                    price={totalPrice}
                    purchaseCancelled={onClosedHandler}
                    purchaseContinued={onPurchaseContinued}
                />
            </Modal>
            <Burger ingredients={ingredints}/>
            <BuildControls 
                purshasable={purshasable}
                price={totalPrice}
                onLessClick={onLessClick} 
                onMoreClick={onMoreClick}
                onOpenModal={onOpenModalHandler}
                ingredients={ingredints}
            />
        </>
    )
}

export default BurgerBuilder;
