'use client';

import { ChangeEvent, useEffect, useState } from 'react';
import { Box, Button, Container, Grid2, Typography } from '@mui/material';
import ArrowDownwardIcon from '@mui/icons-material/ArrowDownward';
import InputField from './components/InputField';
import '@/animations/Loader.css';
import { useAppDispatch, useAppSelector } from '@/store/hooks';
import { Header } from './components/Header/Header';
import { encodeMessage, decodeMessage } from '@/features/requestSlice';

interface IFormData {
   decoded: string;
   encoded: string;
   password: string;
}

export default function Home() {
   const dispatch = useAppDispatch();
   const { loading, result, error } = useAppSelector((state) => state.request);

   const [formData, setFormData] = useState<IFormData>({
      encoded: '',
      decoded: '',
      password: '',
   });

   const onCypherHandler = (): void => {
      if (!formData.password.trim()) {
         alert('Please, type in password!');
         return;
      }

      dispatch(
         encodeMessage({
            password: formData.password,
            message: formData.decoded,
         })
      );
   };

   const onDecypherHandler = (): void => {
      if (!formData.password.trim()) {
         alert('Please, type in password!');
         return;
      }

      dispatch(
         decodeMessage({
            password: formData.password,
            message: formData.encoded,
         })
      );
   };

   useEffect(() => {
      if (result) {
         if (formData.decoded) {
            setFormData((prev) => ({ ...prev, encoded: result }));
         } else {
            setFormData((prev) => ({ ...prev, decoded: result }));
         }
      }
   }, [result]);

   const onInputChangeHandler = (e: ChangeEvent<HTMLInputElement>) => {
      const { name, value } = e.target;
      setFormData((prevData) => ({
         ...prevData,
         [name.toLowerCase()]: value,
      }));
   };

   return (
      <div className="App">
         <Container sx={{ marginTop: 2 }} maxWidth="lg">
            <Header />
            <Box
               sx={{
                  maxWidth: {
                     xs: '90%',
                     sm: '80%',
                     lg: '70%',
                  },
               }}
               component="form"
               autoComplete="off"
               padding={2}
            >
               <Grid2 container direction="column" spacing={2}>
                  <InputField name="Decoded" value={formData.decoded} onChange={onInputChangeHandler} />
                  <Grid2
                     container
                     alignItems="center"
                     sx={{
                        flexDirection: {
                           xs: 'column',
                           md: 'row',
                        },
                     }}
                  >
                     <Grid2 size={7}>
                        <InputField name="Password" value={formData.password} onChange={onInputChangeHandler} />
                     </Grid2>
                     <Grid2 container>
                        <Grid2>
                           <Button size="small" variant="contained" startIcon={<ArrowDownwardIcon />} onClick={onDecypherHandler}>
                              <Typography
                                 sx={{
                                    display: {
                                       xs: 'none',
                                       md: 'inline',
                                    },
                                 }}
                              >
                                 Decode
                              </Typography>
                           </Button>
                        </Grid2>

                        <Grid2>
                           <Button
                              size="small"
                              variant="contained"
                              startIcon={
                                 <ArrowDownwardIcon
                                    sx={{
                                       transform: 'rotate(180deg)',
                                    }}
                                 />
                              }
                              onClick={() => onCypherHandler()}
                           >
                              <Typography
                                 sx={{
                                    display: {
                                       xs: 'none',
                                       md: 'inline',
                                    },
                                 }}
                              >
                                 Encode
                              </Typography>
                           </Button>
                        </Grid2>
                     </Grid2>
                  </Grid2>

                  <InputField name="Encoded" value={formData.encoded} onChange={onInputChangeHandler} />
               </Grid2>
            </Box>

            {loading && (
               <div className="loader">
                  <div className="lds-ripple">
                     <div></div>
                     <div></div>
                  </div>
               </div>
            )}
            {error && <Typography color="error">{error}</Typography>}
         </Container>
      </div>
   );
}
