import { createSlice, createAsyncThunk } from '@reduxjs/toolkit';
import { axiosClient } from '@/helpers/axiosClient';

interface RequestState {
   loading: boolean;
   error?: string | null;
   result: string;
}

interface IData {
   password: string;
   message: string;
}

const initialState: RequestState = {
   loading: false,
   error: null,
   result: '',
};

export const encodeMessage = createAsyncThunk('request/encode', async (newMessage: IData) => {
   try {
      const { data } = await axiosClient.post('/encode', newMessage);

      return data.encoded;
   } catch (error: any) {
      throw new Error(error.response?.data?.message || 'Error decoding message');
   }
});

export const decodeMessage = createAsyncThunk('request/decode', async (newMessage: IData) => {
   try {
      const { data } = await axiosClient.post('/decode', newMessage);
      return data.decoded;
   } catch (error: any) {
      throw new Error(error.response?.data?.message || 'Error encoding message');
   }
});

const requestSlice = createSlice({
   name: 'request',
   initialState,
   reducers: {},
   extraReducers: (builder) => {
      builder
         .addCase(encodeMessage.pending, (state) => {
            state.loading = true;
            state.error = null;
         })
         .addCase(encodeMessage.fulfilled, (state, action) => {
            state.loading = false;
            state.result = action.payload;
         })
         .addCase(encodeMessage.rejected, (state, action) => {
            state.loading = false;
            state.error = action.error.message || 'Error occured';
         })
         .addCase(decodeMessage.pending, (state) => {
            state.loading = true;
            state.error = null;
         })
         .addCase(decodeMessage.fulfilled, (state, action) => {
            state.loading = false;
            state.result = action.payload;
         })
         .addCase(decodeMessage.rejected, (state, action) => {
            state.loading = false;
            state.error = action.error.message || 'Error occured';
         });
   },
});

export default requestSlice.reducer;
