using Microsoft.AspNetCore.Mvc;
using Payment.WebApi.Enums;
using Payment.WebApi.Models.DbModels;
using Payment.WebApi.Models.RequestModels;
using Payment.WebApi.Repositories.Interfaces;

namespace Payment.WebApi.Controllers;

[ApiController]
[Route("Users/[action]")]
public class UserController : Controller
{
    private readonly IUserRepository _userRepository;

    public UserController(IUserRepository userRepository)
    {
        _userRepository = userRepository;
    }

    [HttpGet]
    [ActionName("GetById")]
    public ActionResult<User> GetUser(int id)
    {
        try
        {
            var user = _userRepository.GetFirstOrDefaultById(id);
            if (user is null)
                return NotFound();

            return Ok(user);
        }
        catch (Exception e)
        {
            return StatusCode(500, new { errorMessage = e.Message });
        }
    }

    [HttpPost]
    [ActionName("Create")]
    public ActionResult<DefaultResponse> Create(UserRequestModel requestModel)
    {
        try
        {
            _userRepository.Create(new User
            {
                FirstName = requestModel.FirstName,
                LastName = requestModel.LastName,
                MiddleName = requestModel.MiddleName,
                Age = requestModel.Age,
                Email = requestModel.Email,
                IsActive = true
            });
            _userRepository.Save();
            return Ok(new DefaultResponse
            {
                ResponseStatusType = ResponseStatusType.Ok
            });
        }
        catch (Exception e)
        {
            return StatusCode(500, new { errorMessage = e.Message });
        }
    }

    [HttpPut]
    [ActionName("Edit")]
    public ActionResult<DefaultResponse> Edit(UserRequestModel requestModel)
    {
        try
        {
            var user = _userRepository.GetFirstOrDefaultByEmail(requestModel.Email);
            if (user is null)
                return Ok(new DefaultResponse
                {
                    ResponseStatusType = ResponseStatusType.NotFound
                });
            
            _userRepository.Update(UpdateUser(user, requestModel));
            _userRepository.Save();

            return Ok(new DefaultResponse
            {
                ResponseStatusType = ResponseStatusType.Ok
            });
        }
        catch (Exception e)
        {
            return StatusCode(500, new { errorMessage = e.Message });
        }
    }

    [HttpDelete]
    [ActionName("Delete")]
    public ActionResult<DefaultResponse> Delete(int id)
    {
        var user = _userRepository.GetFirstOrDefaultById(id);
        if (user is null)
            return Ok(new DefaultResponse
            {
                ResponseStatusType = ResponseStatusType.NotFound
            });

        user.IsActive = false;
        _userRepository.Save();
        
        return Ok(new DefaultResponse
        {
            ResponseStatusType = ResponseStatusType.Ok
        });
    }

    private static User UpdateUser(User user, UserRequestModel requestModel)
    {
        user.FirstName = requestModel.FirstName;
        user.LastName = requestModel.LastName;
        user.MiddleName = requestModel.MiddleName;
        user.Age = user.Age;
        return user;
    }
}