using Microsoft.AspNetCore.Mvc;
using Payment.WebApi.Enums;
using Payment.WebApi.Models.DbModels;
using Payment.WebApi.Models.RequestModels;
using Payment.WebApi.Models.RequestModels.Users;
using Payment.WebApi.Repositories.Interfaces;
using Swashbuckle.AspNetCore.Annotations;

namespace Payment.WebApi.Controllers;

[ApiController]
[Route("Users/[action]")]
[Produces("application/json")]
public class UserController : Controller
{
    private readonly IUserRepository _userRepository;

    public UserController(IUserRepository userRepository)
    {
        _userRepository = userRepository;
    }

    [HttpGet]
    [ActionName("GetById")]
    [ProducesResponseType(200)]
    [ProducesResponseType(204)]
    [ProducesResponseType(500)]
    [SwaggerOperation
        (
            Description = "Получение данных по пользователю по его айди. Если пользователь не найден, то возвращается 204 статус",
            OperationId = "GetByUserId",
            Tags = new []{"Users"}
        )
    ]
    public ActionResult<UserResponseModel> GetUser(int id)
    {
        try
        {
            var user = _userRepository.GetFirstOrDefaultById(id);
            if (user is null)
                return NoContent();

            return Ok(UserMap(user));
        }
        catch (Exception e)
        {
            return StatusCode(500, new { errorMessage = e.Message });
        }
    }

    [HttpPost]
    [ActionName("Create")]
    [ProducesResponseType(200)]
    [ProducesResponseType(400)]
    [ProducesResponseType(500)]
    [SwaggerOperation
        (
            Description = "Создание пользователя",
            OperationId = "CreateUser",
            Tags = new []{"Users"}
        )
    ]
    public ActionResult<DefaultResponse> Create(UserCreateRequestModel createRequestModel)
    {
        try
        {
            _userRepository.Create(new User
            {
                FirstName = createRequestModel.FirstName,
                LastName = createRequestModel.LastName,
                MiddleName = createRequestModel.MiddleName,
                Age = createRequestModel.Age,
                Email = createRequestModel.Email,
                IsActive = true
            });
            _userRepository.Save();
            return Ok(new DefaultResponse
            {
                ResponseStatusType = ResponseStatusType.Ok
            });
        }
        catch (Exception e)
        {
            return StatusCode(500, new { errorMessage = e.Message });
        }
    }

    [HttpPut]
    [ActionName("Edit")]
    [ProducesResponseType(200)]
    [ProducesResponseType(500)]
    [SwaggerOperation
        (
            Description = "Редактирование данных пользователя, пока что без почты",
            OperationId = "UpdateUser",
            Tags = new []{"Users"}
        )
    ]
    public ActionResult<DefaultResponse> Edit(UserEditRequestModel createRequestModel)
    {
        try
        {
            var user = _userRepository.GetFirstOrDefaultByEmail(createRequestModel.Email);
            if (user is null)
                return Ok(new DefaultResponse
                {
                    ResponseStatusType = ResponseStatusType.NotFound
                });

            _userRepository.Update(UpdateUser(user, createRequestModel));
            _userRepository.Save();

            return Ok(new DefaultResponse
            {
                ResponseStatusType = ResponseStatusType.Ok
            });
        }
        catch (Exception e)
        {
            return StatusCode(500, new { errorMessage = e.Message });
        }
    }

    [HttpDelete]
    [ActionName("Delete")]
    [ProducesResponseType(200)]
    [ProducesResponseType(500)]
    [SwaggerOperation
        (
            Description = "Деактивация пользователя. К деактивированным пользователям имеет доступ только администратор системы",
            OperationId = "DeleteUser",
            Tags = new []{"Users"}
        )
    ]
    public ActionResult<DefaultResponse> Delete(int id)
    {
        var user = _userRepository.GetFirstOrDefaultById(id);
        if (user is null)
            return Ok(new DefaultResponse
            {
                ResponseStatusType = ResponseStatusType.NotFound
            });

        user.IsActive = false;
        _userRepository.Save();

        return Ok(new DefaultResponse
        {
            ResponseStatusType = ResponseStatusType.Ok
        });
    }

    private static User UpdateUser(User user, UserEditRequestModel createRequestModel)
    {
        user.FirstName = createRequestModel.FirstName;
        user.LastName = createRequestModel.LastName;
        user.MiddleName = createRequestModel.MiddleName;
        user.Age = user.Age;
        return user;
    }

    private static UserResponseModel UserMap(User user)
    {
        return new UserResponseModel
        {
            FirstName = user.FirstName,
            LastName = user.LastName,
            MiddleName = user.MiddleName,
            Age = user.Age,
            Email = user.Email,
            Balances = user.Balances.Select(s => new UserBalancesResponseModel
            {
                Currency = s.Currency,
                Amount = s.Amount
            }).ToList()
        };
    }
}